﻿namespace SampleApp.Services
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Newtonsoft.Json;
    using Newtonsoft.Json.Linq;
    using RestSharp;

    /// <summary>
    /// Service to execute Public API related requests.
    /// </summary>
    public class ApiService
    {
        private const string DefaultResponseContent = "{ Success: false }";

        private IRestClient client;
        private AuthenticationService authService;
        private ILog logger;

        public ApiService(IRestClient client, AuthenticationService authService, ILog logger)
        {
            this.client = client;
            this.authService = authService;
            this.logger = logger;
        }

        /// <summary>
        /// Executes GET request to the public endpoint.
        /// </summary>
        public async Task<T> Get<T>(IRestRequest request)
            where T : DTO.ISuccess
        {
            var model = await Execute<T>(() => client.ExecuteGetAsync(request));

            if (!model.Success)
            {
                throw new ApiException(model.Message);
            }

            return model;
        }

        /// <summary>
        /// Executes parameterless GET request to the public endpoint.
        /// </summary>
        public Task<T> Get<T>(string path)
            where T : DTO.ISuccess
        {
            return Get<T>(new RestRequest(path));
        }

        /// <summary>
        /// Generates new data request for the object collection provided.
        /// </summary>
        public IRestRequest CreateDataRequest<T>(string path, ICollection<DTO.DataObjectLite> collection)
            where T : DTO.IDataObjectModel, new()
        {
            var req = new T();
            req.DataCollection = new DTO.DataObjectLiteCollection();
            req.DataCollection.Rows = collection;
            return new RestRequest(path).AddJsonBody(req);
        }

        /// <summary>
        /// Generates new data request for the object collection provided.
        /// </summary>
        public IRestRequest CreateGenericDataRequest(string path, ICollection<DTO.DataObjectLite> collection, string objectId, string ProcessId)
        {
            var req = new DTO.GenericDataObjectCreateRequest();
            req.DataCollection = new DTO.DataObjectLiteCollection();
            req.DataCollection.Rows = collection;
            req.ObjectId = objectId;
            req.ProcessId = ProcessId;
            return new RestRequest(path).AddJsonBody(req);
        }

        /// <summary>
        /// Executes POST request to the public endpoint.
        /// </summary>
        public async Task<T> Create<T>(IRestRequest request)
            where T : DTO.ISuccessModel
        {
            var model = await Execute<T>(() => client.ExecutePostAsync(request));

            if (!model.Success)
            {
                throw new ApiDataException(model.Message) { DataCollection = model.DataCollection };
            }

            return model;
        }

        /// <summary>
        /// Returns the Matter list for the client provided.
        /// </summary>
        public async Task<ICollection<DTO.Matter>> GetMattersByClient(int clientIndex)
        {
            var res = await Get<DTO.MatterGetResponse>(new RestRequest("matter")
                .AddParameter("AdvancedFilter.Filter.Predicates[0].Attribute", "Client")
                .AddParameter("AdvancedFilter.Filter.Predicates[0].Operator", "IsEqualTo")
                .AddParameter("AdvancedFilter.Filter.Predicates[0].Value", clientIndex));

            return res.Matters;
        }

        /// <summary>
        /// Returns the Matter list for the IDs specified.
        /// </summary>
        public async Task<ICollection<DTO.Matter>> GetMattersByIDs(Guid[] ids)
        {
            var req = new RestRequest("matter")
                .AddParameter("AdvancedFilter.Filter.Predicates[0].Attribute", "MatterID")
                .AddParameter("AdvancedFilter.Filter.Predicates[0].Operator", "IsIn");

            for (int i = 0; i < ids.Length; i++)
            {
                req.AddParameter($"AdvancedFilter.Filter.Predicates[0].Value[{i}]", ids[i]);
            }

            var res = await Get<DTO.MatterGetResponse>(req);
            return res.Matters;
        }

        /// <summary>
        /// Returns the Matter by the Matter Number.
        /// </summary>
        public async Task<DTO.Matter> GetMatter(string number)
        {
            var res = await Get<DTO.MatterGetResponse>(
                new RestRequest("matter").AddParameter("Number", number));

            if (res.Matters.Count == 0)
            {
                throw new ApiException($"Matter(Number={number}) cannot be found.");
            }

            return res.Matters.First();
        }

        /// <summary>
        /// Returns the Client by the Client Number.
        /// </summary>
        public async Task<DTO.Client> GetClient(string number)
        {
            var res = await Get<DTO.ClientGetResponse>(
                new RestRequest("client").AddParameter("Number", number));

            if (res.Clients.Count == 0)
            {
                throw new ApiException($"Client(Number={number}) cannot be found.");
            }

            return res.Clients.First();
        }

        /// <summary>
        /// Returns the Client by the Client Index.
        /// </summary>
        public async Task<DTO.Client> GetClient(int index)
        {
            var res = await Get<DTO.ClientGetResponse>(
                new RestRequest("client").AddParameter("ClientIndex", index));

            if (res.Clients.Count == 0)
            {
                throw new ApiException($"Client(ClientIndex={index}) cannot be found.");
            }

            return res.Clients.First();
        }

        /// <summary>
        /// Returns the Client by the Client ID.
        /// </summary>
        public async Task<DTO.Client> GetClient(Guid itemId)
        {
            var res = await Get<DTO.ClientGetResponse>(
                new RestRequest("client").AddParameter("ClientId", itemId));

            if (res.Clients.Count == 0)
            {
                throw new ApiException($"Client(ClientId={itemId}) cannot be found.");
            }

            return res.Clients.First();
        }

        /// <summary>
        /// Returns the Entity by the EntIndex.
        /// </summary>
        public async Task<DTO.Entity> GetEntity(int index)
        {
            var res = await Get<DTO.EntityGetResponse>(
                new RestRequest("entity/all").AddParameter("EntityIndex", index));

            if (res.Organizations.Any())
            {
                return res.Organizations.First();
            }

            if (res.Persons.Any())
            {
                return res.Persons.First();
            }

            throw new ApiException($"Entity(EntIndex={index}) cannot be found.");
        }

        /// <summary>
        /// Returns the Entity by the Entity ID.
        /// </summary>
        public async Task<DTO.Entity> GetEntity(Guid itemId)
        {
            var res = await Get<DTO.EntityGetResponse>(
                new RestRequest("entity/all").AddParameter("EntityId", itemId));

            if (res.Organizations.Any())
            {
                return res.Organizations.First();
            }

            if (res.Persons.Any())
            {
                return res.Persons.First();
            }

            throw new ApiException($"Entity(EntityId={itemId}) cannot be found.");
        }

        /// <summary>
        /// Returns the Person by the EntIndex.
        /// </summary>
        public async Task<DTO.Entity> GetEntityPerson(int index)
        {
            var res = await Get<DTO.EntityGetResponse>(
                new RestRequest("entity/person").AddParameter("EntityIndex", index));

            if (res.Persons.Count == 0)
            {
                throw new ApiException($"Person(EntIndex={index}) cannot be found.");
            }

            return res.Persons.First();
        }

        /// <summary>
        /// Returns the Person by the Person ID.
        /// </summary>
        public async Task<DTO.Entity> GetEntityPerson(Guid itemId)
        {
            var res = await Get<DTO.EntityGetResponse>(
                new RestRequest("entity/person").AddParameter("EntityId", itemId));

            if (res.Persons.Count == 0)
            {
                throw new ApiException($"Person(EntityId={itemId}) cannot be found.");
            }

            return res.Persons.First();
        }

        /// <summary>
        /// Returns the Organization by the EntIndex.
        /// </summary>
        public async Task<DTO.Entity> GetEntityOrg(int index)
        {
            var res = await Get<DTO.EntityGetResponse>(
                new RestRequest("entity/organization").AddParameter("EntityIndex", index));

            if (res.Organizations.Count == 0)
            {
                throw new ApiException($"Organization(EntIndex={index}) cannot be found.");
            }

            return res.Organizations.First();
        }

        /// <summary>
        /// Returns the Timekeeper by the Timekeeper Number.
        /// </summary>
        public async Task<DTO.Timekeeper> GetTimekeeper(string number)
        {
            var res = await Get<DTO.TimekeeperGetResponse>(
                new RestRequest("timekeeper").AddParameter("Number", number));

            if (res.Timekeepers.Count == 0)
            {
                throw new ApiException($"Timekeeper(Number={number}) cannot be found.");
            }

            return res.Timekeepers.First();
        }

        /// <summary>
        /// Returns the Timecard by the Time Index.
        /// </summary>
        public async Task<DTO.Timecard> GetTimecard(int index)
        {
            var res = await Get<DTO.TimecardGetResponse>(
                new RestRequest("time/posted").AddParameter("TimeIndex", index));

            if (res.Timecards.Count == 0)
            {
                throw new ApiException($"TimeCard(TimeIndex={index}) cannot be found.");
            }

            return res.Timecards.First();
        }

        /// <summary>
        /// Returns the Voucher by the Voucher Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetVoucher(int index)
        {
            var res = await Get<DTO.VoucherGetResponse>(
                new RestRequest("dataobject?ObjectId=Voucher&ProcessId=Voucher").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"Voucher(Voucher={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }

        /// <summary>
        /// Returns the ChargeCard by the Charge Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetChargecard(int index)
        {
            var res = await Get<DTO.ChargeCardGetResponse>(
                new RestRequest("dataobject?ObjectId=ChrgCard&ProcessId=ChrgCardUpdate").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"ChargeCard(ChargeCard={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }

        /// <summary>
        /// Returns the VendorPayee by the VendorPayee Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetVendorPayee(int index)
        {
            var res = await Get<DTO.VendorPayeeGetResponse>(
                new RestRequest("dataobject?ObjectId=NewVendorPayee&ProcessId=NewVendorPayee").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"NewVendorPayee(NewVendorPayee={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }

        /// <summary>
        /// Returns the VendorPayee by the VendorPayee Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetDirectCheck(int index)
        {
            var res = await Get<DTO.DirectCheckGetResponse>(
                new RestRequest("dataobject?ObjectId=CkDirect&ProcessId=DirectCk").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"NewDirectCheck(DirectCheck={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }


        /// <summary>
        /// Returns the Payee by the Payee Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetPayee(int index)
        {
            var res = await Get<DTO.PayeeGetResponse>(
                new RestRequest("dataobject?ObjectId=Payee&ProcessId=PayeeMnt").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"Payee(Payee={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }

        /// <summary>
        /// Returns the ChargeCardPending by the Charge Index.
        /// </summary>
        public async Task<DTO.DataObjectLite> GetChargecardPending(int index)
        {
            var res = await Get<DTO.ChargeCardGetResponse>(
                new RestRequest("dataobject?ObjectId=ChrgCardPending&ProcessId=ChrgCardPending").AddParameter("Key", index));

            if (res.DataCollection.Rows.Count == 0)
            {
                throw new ApiException($"ChargeCardPending(ChargeCardPending={index}) cannot be found.");
            }
            return res.DataCollection.Rows.First();
        }
        public async Task<DTO.TimecardPending> GetTimecardPending(int index)
        {
            var res = await Get<DTO.TimecardPendingGetResponse>(
                new RestRequest("time/pending").AddParameter("TimePendIndex", index));

            if (res.Timecards.Count == 0)
            {
                throw new ApiException($"TimeCardPending(TimePendIndex={index}) cannot be found.");
            }

            return res.Timecards.First();
        }

        /// <summary>
        /// Returns the CostCard by the Cost Index.
        /// </summary>
        public async Task<DTO.CostCard> GetCostcard(int index)
        {
            var res = await Get<DTO.CostCardGetResponse>(
                new RestRequest("cost/posted").AddParameter("CostIndex", index));

            if (res.CostCards.Count == 0)
            {
                throw new ApiException($"CostCard(CostIndex={index}) cannot be found.");
            }

            return res.CostCards.First();
        }

        /// <summary>
        /// Returns the CostCardPending by the Cost Index.
        /// </summary>
        public async Task<DTO.CostCardPending> GetCostcardPending(int index)
        {
            var res = await Get<DTO.CostCardPendingGetResponse>(
                new RestRequest("cost/pending").AddParameter("CostPeindIndex", index));

            if (res.CostCards.Count == 0)
            {
                throw new ApiException($"CostCard(CostPeindIndex={index}) cannot be found.");
            }

            return res.CostCards.First();
        }

        private async Task<T> Execute<T>(Func<Task<IRestResponse>> func)
            where T : DTO.ISuccess
        {
            var response = await func();

            // Try to re-authenticate in the case of 401 error response and resend the request if succeed.
            // Coudld happen when token is expired or if current(Windows) user is not recognized by remote host.
            while (response.StatusCode == System.Net.HttpStatusCode.Unauthorized
                && await this.authService.Authenticate())
            {
                response = await func();
            }

            return ProcessResponse<T>(response);
        }

        private T ProcessResponse<T>(IRestResponse response)
            where T : DTO.ISuccess
        {
            if (response.ResponseStatus != ResponseStatus.Completed)
            {
                // Not able to connect the server.
                logger.Log($"[{response.Request.Method}] {response.Request.Resource}: {response.ResponseStatus})");
                logger.Log(response.ErrorMessage);
                return CreateDefaultResponse<T>();
            }
            else
            {
                logger.Log($"[{response.Request.Method}] {response.Request.Resource}: {(int)response.StatusCode} ({response.StatusDescription})");
                Log3EMessage(response);

                if (response.IsSuccessful
                    || response.StatusCode == System.Net.HttpStatusCode.BadRequest
                    || response.StatusCode == System.Net.HttpStatusCode.NotFound)
                {
                    // Return Public API success model based response.
                    var model = JsonConvert.DeserializeObject<T>(response.Content);
                    return model;                 
                }

                // Something went globally wrong (ex. Internal Server Error).
                // Most likely response does not contain a valid JSON body in such case.
                // Just log the content as it is and return defauld failure model. 
                logger.Log(response.Content);
                return CreateDefaultResponse<T>();
            }
        }

        private T CreateDefaultResponse<T>()
             where T : DTO.ISuccess
        {
            return JsonConvert.DeserializeObject<T>(DefaultResponseContent);
        }

        /// <summary>
        /// Logs standard 3E Message header if provided.
        /// </summary>
        private void Log3EMessage(IRestResponse response)
        {
            foreach (var header in response.Headers)
            {
                if (header.Name == "X-3E-Message")
                {
                    logger.Log("X-3E-Message: " + Uri.UnescapeDataString(header.Value.ToString()));
                    break;
                }
            }
        }
    }
}
