﻿namespace SampleApp.Services
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Newtonsoft.Json;

    /// <summary>
    /// Service to provide basic operations related to DataObjectLite object manipulations.
    /// </summary>
    public class DataObjectService
    {
        private static HashSet<string> NxItemAttributes = new HashSet<string>(StringComparer.OrdinalIgnoreCase) {
            "ItemID", "ArchetypeCode", "CurrProcItemID", "LastProcItemID", "OrigProcItemID", "HasAttachments", "TimeStamp"
        };

        /// <summary>
        /// Returns the Data Object parsed.
        /// </summary>
        public DTO.DataObjectLite Parse(string model)
        {
            return JsonConvert.DeserializeObject<DTO.DataObjectLite>(model);
        }

        /// <summary>
        /// Returns the list of Data Objects parsed.
        /// </summary>
        public DTO.DataObjectLite[] ParseMany(string model)
        {
            return JsonConvert.DeserializeObject<DTO.DataObjectLite[]>(model);
        }

        /// <summary>
        /// Creates full copy of the source Data Object keeping only required attributes.
        /// </summary>
        public DTO.DataObjectLite ModelFrom(DTO.DataObjectLite source)
        {
            return Transform(source, (id, attr) =>
            {
                if ((int)attr.AccessType > 0 || NxItemAttributes.Contains(id))
                {
                    return null;
                }

                var updated = new DTO.Attribute();
                updated.Value = attr.Value ?? string.Empty;
                return updated;
            });
        }

        /// <summary>
        /// Creates full copy of the source Data Object filtering out NxItem and related(ex. ClientRel.Description) attributes.
        /// </summary>
        public DTO.DataObjectLite Clone(DTO.DataObjectLite source)
        {
            return Transform(source, (id, attr) =>
            {
                if (NxItemAttributes.Contains(id) || id.Contains('.'))
                {
                    return null;
                }

                var updated = new DTO.Attribute();
                updated.Value = attr.Value ?? string.Empty;
                return updated;
            });
        }

        /// <summary>
        /// Selects a list of all child objects corresponded to the children schema path(ex. 'Relate/Site').
        /// </summary>
        public IList<DTO.DataObjectLite> Select(DTO.DataObjectLite source, string path)
        {
            var found = new List<DTO.DataObjectLite>();
            Select(source, path, found);
            return found;
        }

        /// <summary>
        /// Creates full copy of the source Data Object provided applying the format function to its attributes.
        /// </summary>
        public DTO.DataObjectLite Generate(DTO.DataObjectLite model, Func<string, string> format)
        {
            return Transform(model, (id, attr) =>
            {
                var updated = new DTO.Attribute();

                if (attr.Value != null)
                {
                    updated.Value = format(attr.Value);
                }

                if (attr.AliasValue != null)
                {
                    updated.AliasValue = format(attr.AliasValue);
                }

                return updated;
            });
        }

        /// <summary>
        /// Returns the Data Object ItemID value.
        /// </summary>
        public Guid GetId(DTO.DataObjectLite model)
        {
            return Guid.Parse(model.Id);
        }

        /// <summary>
        /// Returns the Data Object ItemID value collection for the range specified.
        /// </summary>
        public Guid[] GetIds(IList<DTO.DataObjectLite> data, int index, int count)
        {
            var ids = new Guid[count];

            for (int i = 0; i < ids.Length; i++)
            {
                ids[i] = GetId(data[index + i]);
            }

            return ids;
        }

        /// <summary>
        /// Returns the Data Object attribute value.
        /// </summary>
        public string Get(DTO.DataObjectLite model, string attributeId, string defaultValue = null)
        {
            if (model.Attributes.TryGetValue(attributeId, out var attribute))
            {
                return attribute.Value;
            }

            return defaultValue;
        }

        /// <summary>
        /// Returns the Data Object attribute value converted to Int32.
        /// </summary>
        public int GetInt32(DTO.DataObjectLite model, string attributeId)
        {
            return int.Parse(Get(model, attributeId));
        }

        /// <summary>
        /// Extends the Data Object attribute value with string provided.
        /// </summary>
        public void Append(DTO.DataObjectLite model, string attributeId, string value)
        {
            var attribute = EnsureAttribute(model, attributeId);
            attribute.Value = string.IsNullOrEmpty(attribute.Value) ? value : attribute.Value + " " + value;
        }

        /// <summary>
        /// Removes from the Data Object the attribte list specified.
        /// </summary>
        public void Remove(DTO.DataObjectLite model, params string[] attributes)
        {
            foreach (var attribute in attributes)
            {
                model.Attributes.Remove(attribute);
            }
        }

        /// <summary>
        /// Updates the Data Object attribute value.
        /// </summary>
        public void Set(DTO.DataObjectLite model, string attributeId, string value)
        {
            var attribute = EnsureAttribute(model, attributeId);
            attribute.Value = value;
        }

        public DTO.DataObjectLite Transform(DTO.DataObjectLite source, Func<string, DTO.Attribute, DTO.Attribute> map)
        {
            var target = new DTO.DataObjectLite();
            target.SubclassId = source.SubclassId;
            target.Attributes = new Dictionary<string, DTO.Attribute>();

            if (source.Attributes != null)
            {
                foreach (var attribute in source.Attributes)
                {
                    var targetAttr = map(attribute.Key, attribute.Value);

                    if (targetAttr != null)
                    {
                        target.Attributes.Add(attribute.Key, targetAttr);
                    }
                }
            }

            target.ChildObjects = new List<DTO.DataObjectLiteCollection>();

            foreach (var sourceChild in GetChildObjects(source))
            {
                var targetChild = Transform(sourceChild, map);

                if (targetChild != null && targetChild.Rows.Any())
                {
                    target.ChildObjects.Add(targetChild);
                }
            }

            return target;
        }

        public DTO.DataObjectLiteCollection Transform(DTO.DataObjectLiteCollection source, Func<string, DTO.Attribute, DTO.Attribute> map)
        {
            var target = new DTO.DataObjectLiteCollection();
            target.Id = source.Id;
            target.ObjectId = source.ObjectId;
            target.Rows = new List<DTO.DataObjectLite>();

            foreach (var obj in source.Rows)
            {
                target.Rows.Add(Transform(obj, map));
            }

            return target;
        }

        private IEnumerable<DTO.DataObjectLiteCollection> GetChildObjects(DTO.DataObjectLite source)
        {
            if (source.ChildObjects == null)
            {
                return Enumerable.Empty<DTO.DataObjectLiteCollection>();
            }

            // Filter out well-known readonly relationships.
            if (source.SubclassId.StartsWith("Entity", StringComparison.OrdinalIgnoreCase))
            {
                return source.ChildObjects.Where(ch => !string.Equals("EntityUse", ch.Id, StringComparison.OrdinalIgnoreCase));
            }

            return source.ChildObjects;
        }

        private void Select(DTO.DataObjectLite source, string path, IList<DTO.DataObjectLite> found)
        {
            var separator = path.IndexOf('/');

            if (separator >= 0)
            {
                var current = path.Substring(0, separator);
                var next = path.Substring(separator + 1);

                ForEach(source.ChildObjects, current, item => Select(item, next, found));
            }
            else
            {
                ForEach(source.ChildObjects, path, item => found.Add(item));
            }
        }

        private void ForEach(ICollection<DTO.DataObjectLiteCollection> collections, string id, Action<DTO.DataObjectLite> action)
        {
            var collection = collections.FirstOrDefault(ch => string.Equals(ch.Id, id, StringComparison.OrdinalIgnoreCase));

            if (collection != null)
            {
                foreach (var item in collection.Rows)
                {
                    action(item);
                }
            }
        }

        private DTO.Attribute EnsureAttribute(DTO.DataObjectLite model, string attributeId)
        {
            if (!model.Attributes.TryGetValue(attributeId, out var attribute))
            {
                attribute = new DTO.Attribute();
                model.Attributes.Add(attributeId, attribute);
            }

            return attribute;
        }
    }
}
