﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Net;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using IdentityModel.OidcClient.Browser;
using SampleApp.Services;

namespace SampleApp.Views
{
    /// <summary>
    /// Browser based login view for CIAM authentication.
    /// </summary>
    public partial class LoginView : UserControl, IBrowser, IView
    {
        private ConfigurationService configurationService;
        private CiamSettings settings;
        private HttpListener listener;

        public LoginView(ConfigurationService configurationService)
        {
            this.configurationService = configurationService;
            this.settings = this.configurationService.GetCiamSettings();

            InitializeComponent();
        }

        public string Caption => "Login";

        /// <summary>
        /// Navigates to the CIAM authentication start URL while waiting for the response on the local port.
        /// </summary>
        public async Task<BrowserResult> InvokeAsync(BrowserOptions options, CancellationToken cancellationToken = default)
        {
            StartListener();

            try
            {
                webBrowser1.Navigate(options.StartUrl);
                var authCode = await WaitForAuthResponse();

                return new BrowserResult
                {
                    Response = authCode,
                    ResultType = BrowserResultType.Success
                };
            }
            catch (Exception ex)
            {
                return new BrowserResult
                {
                    ResultType = BrowserResultType.UnknownError,
                    Error = ex.Message
                };
            }
            finally
            {
                StopListener();
            }
        }

        private void StartListener()
        {
            listener = new HttpListener();
            listener.Prefixes.Add($"http://localhost:{settings.ReplyPort}/");
            listener.Start();
        }

        private void StopListener()
        {
            listener.Stop();
        }

        private async Task<string> WaitForAuthResponse()
        {
            var context = await listener.GetContextAsync();

            if (context.Request.HttpMethod == "GET" && context.Request.Url.AbsolutePath.EndsWith(settings.ReplyPath))
            {
                context.Response.StatusCode = 200;
                context.Response.Close();
                return context.Request.Url.Query;
            }

            context.Response.StatusCode = 403;
            context.Response.Close();
            throw new Exception("Invalid authentication response.");
        }
    }
}
