﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using RestSharp;
using SampleApp;
using SampleApp.Controls;
using SampleApp.Services;
using SampleApp.Views;

public class CostCardPendingView : CaseBaseView
{
    CostcardPendingInput input;

    public CostCardPendingView(IRestClient client, AuthenticationService authService)
        : base(client, authService)
    {
        this.SetLookupName("Cost Index");
        this.input = new CostcardPendingInput(this, this.objectService, this.cacheService);
        this.AddInput(input);
        this.Reset();
    }

    public override string Caption => "CostCardPending";

    protected override void Reset()
    {
        var template = templateService.GetTemplate(TemplateId.CostCardPending);
        input.SetCostCardTemplate(template);
    }

    protected async override Task Lookup(string index)
    {
        if (!int.TryParse(index, out var costIndex) || costIndex <= 0)
        {
            Log("Please enter a valid Cost Index to copy from.");
            return;
        }

        var costcard = await apiService.GetCostcardPending(costIndex);
        var model = Clone(costcard.AsObjectLite());
        var template = templateService.GetTemplate(TemplateId.CostCardPending, model);
        input.SetCostCardTemplate(template);
    }

    /// <summary>
    /// Generates CostCard list.
    /// </summary>
    protected override async Task Execute()
    {
        if (input.TryGetParameters(out var parameters))
        {
            var costcards = new SampleApp.DTO.DataObjectLite[parameters.NumToCreate];
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);
            var matterIndex = 0;

            for (var i = 0; i < costcards.Length; i++)
            {
                var date = parameters.StartDate.AddDays(i / parameters.CostCardsPerDay).ToString("yyyy-MM-dd");
                costcards[i] = Generate(parameters.CostCardPending, CreateFormatter(suffix.Increment(), date));

                if (parameters.Matters != null)
                {
                    Set(costcards[i], "Matter", parameters.Matters[matterIndex++]);
                    matterIndex %= parameters.Matters.Count;
                }
            }

            await apiService.Create<SampleApp.DTO.CostCardCreateResponse>(
                apiService.CreateDataRequest<SampleApp.DTO.CostCardCreateRequest>("cost/pending", costcards));

            Log($"Total {parameters.NumToCreate} CostCard(s) successfully created.");
        }
    }
}