﻿namespace SampleApp.Services
{
    using System;
    using System.Collections.Generic;
    using System.CodeDom.Compiler;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using RestSharp;

    /// <summary>
    /// Service to store and generate text representation for DataObjectLite templates.
    /// </summary>
    public class TemplateService
    {
        public const string SuffixToken = "%suffix%";
        public const string DateToken = "%date%";

        private const string DefaultTemplate = "{}";
        private const int TemplateCount = 15;
        private static string[] templates;

        private ApiService client;
        private ILog logger;
        private DataObjectService objectService;

        public TemplateService(IRestClient client, AuthenticationService authService, ILog logger)
        {
            this.client = new ApiService(client, authService, logger);
            this.logger = logger;
            this.objectService = new DataObjectService();
        }

        /// <summary>
        /// Loads default template list.
        /// </summary>
        /// <returns></returns>
        public async Task LoadTemplates()
        {
            logger.Log("Loading Data Object templates...");

            templates = new string[TemplateCount];
            templates[(int)TemplateId.Timekeeper] = await GetTimekeeperTemplate();
            templates[(int)TemplateId.Matter] = await GetMatterTemplate();
            templates[(int)TemplateId.Client] = await GetClientTemplate();
            templates[(int)TemplateId.EntityOrganization] = await GetEntityOrgTemplate();
            templates[(int)TemplateId.EntityPerson] = await GetEntityPersonTemplate();
            templates[(int)TemplateId.TimeCard] = await GetTimecardTemplate();
            templates[(int)TemplateId.CostCard] = await GetCostcardTemplate();
            templates[(int)TemplateId.CostCardPending] = await GetCostcardPendingTemplate();
            templates[(int)TemplateId.TimeCardPending] = await GetTimecardPendingTemplate();
            templates[(int)TemplateId.Voucher] = await GetVoucherTemplate();
            templates[(int)TemplateId.ChargeCard] = await GetChargeCardTemplate();
            templates[(int)TemplateId.ChargeCardPending] = await GetChargeCardPendingTemplate();
            templates[(int)TemplateId.Payee] = await GetPayeeTemplate();
            templates[(int)TemplateId.NewVendorPayee] = await GetVendorPayeeTemplate();
            templates[(int)TemplateId.DirectCheck] = await GetDirectCheckTemplate();
        }

        /// <summary>
        /// Returns default template for the entity type specified.
        /// </summary>
        public string GetTemplate(TemplateId id)
        {
            return templates[(int)id] ?? DefaultTemplate;
        }

        /// <summary>
        /// Generates new template based on the model provided.
        /// </summary>
        public string GetTemplate(TemplateId id, DTO.DataObjectLite model)
        {
            return Format(ConfigureTemplate(id, model));
        }

        /// <summary>
        /// Generates new template based on the model collection provided.
        /// </summary>
        public string GetTemplateList(TemplateId id, IEnumerable<DTO.DataObjectLite> models)
        {
            return Format(models.Select(model => ConfigureTemplate(id, model)));
        }

        /// <summary>
        /// Updates the model attributes according to the entity type specified.
        /// </summary>
        public DTO.DataObjectLite ConfigureTemplate(TemplateId id, DTO.DataObjectLite model)
        {
            switch (id)
            {
                case TemplateId.Timekeeper:
                    objectService.Remove(model, "TkprIndex", "Number");
                    objectService.Append(model, "DisplayName", SuffixToken);
                    break;
                case TemplateId.Matter:
                    objectService.Remove(model, "MattIndex", "Number");
                    objectService.Append(model, "DisplayName", SuffixToken);
                    objectService.Append(model, "Description", SuffixToken);
                    break;
                case TemplateId.Client:
                    objectService.Remove(model, "ClientIndex", "Number");
                    objectService.Append(model, "DisplayName", SuffixToken);
                    objectService.Append(model, "Description", SuffixToken);
                    break;
                case TemplateId.EntityOrganization:
                    objectService.Remove(model, "EntIndex");
                    objectService.Append(model, "OrgName", SuffixToken);
                    break;
                case TemplateId.EntityPerson:
                    objectService.Remove(model, "EntIndex");
                    objectService.Append(model, "LastName", SuffixToken);
                    break;
                case TemplateId.TimeCard:
                    objectService.Remove(model, "TimeIndex", "EBillValMessage", "BillingViolations");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.TimeCardPending:
                    objectService.Remove(model, "TimeIndex", "EBillValMessage", "BillingViolations", "PsuedoMatter", "PsuedoDesc");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.CostCard:
                    objectService.Remove(model, "CostIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.CostCardPending:
                    objectService.Remove(model, "CostIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.Voucher:
                    objectService.Remove(model, "ChildObjects","GLAcct.GLUnit", "GLAcct.GLNatural", "GLAcct.GLOffice", "GLAcct.GLDepartment", "GLAcct.GLActivity");
                    objectService.Set(model, "InvDate", DateToken);
                    objectService.Append(model, "InvNum", SuffixToken);
                    break;
                case TemplateId.ChargeCard:
                    objectService.Remove(model, "ChargeIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.ChargeCardPending:
                    objectService.Remove(model, "ChargeIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.Payee:
                    objectService.Remove(model, "PayeeIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    objectService.Append(model, "Name", SuffixToken);
                    break;
                case TemplateId.NewVendorPayee:
                    objectService.Remove(model, "VendorIndex");
                    objectService.Set(model, "WorkDate", DateToken);
                    break;
                case TemplateId.DirectCheck:
                    objectService.Remove(model, "CkIndex", "DCSOutputFldr");
                    objectService.Set(model, "WorkDate", DateToken);
                    objectService.Append(model, "CkNum", SuffixToken);
                    break;
            }

            return model;
        }

        private async Task<string> GetTimekeeperTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("timekeeper/template", TemplateId.Timekeeper);
        }

        private async Task<string> GetMatterTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("matter/template", TemplateId.Matter);
        }

        private async Task<string> GetClientTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("client/template", TemplateId.Client);
        }

        private async Task<string> GetEntityOrgTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("entity/organization/template", TemplateId.EntityOrganization);
        }

        private async Task<string> GetEntityPersonTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("entity/person/template", TemplateId.EntityPerson);
        }

        private async Task<string> GetTimecardTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("time/posted/template", TemplateId.TimeCard);
        }

        private async Task<string> GetCostcardTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("cost/posted/template", TemplateId.CostCard);
        }

        private async Task<string> GetCostcardPendingTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("cost/pending/template", TemplateId.CostCardPending);
        }

        private async Task<string> GetTimecardPendingTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("time/pending/template", TemplateId.TimeCardPending);
        }

        private async Task<string> GetVoucherTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId="+ TemplateId.Voucher + "&ProcessId="+ TemplateId.Voucher + "", TemplateId.Voucher);
        }

        private async Task<string> GetChargeCardTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId=ChrgCard&ProcessId=ChrgCardUpdate", TemplateId.ChargeCard);
        }

        private async Task<string> GetChargeCardPendingTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId=ChrgCardPending&ProcessId=ChrgCardPending", TemplateId.ChargeCardPending);
        }

        private async Task<string> GetPayeeTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId=Payee&ProcessId=PayeeMnt", TemplateId.Payee);
        }

        private async Task<string> GetVendorPayeeTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId=NewVendorPayee&ProcessId=NewVendorPayee", TemplateId.NewVendorPayee);
        }

        private async Task<string> GetDirectCheckTemplate()
        {
            return await GetTemplate<DTO.TimekeeperTemplateResponse>("dataobject/template?ObjectId=CkDirect&ProcessId=DirectCk", TemplateId.NewVendorPayee);
        }
        private async Task<string> GetTemplate<T>(string path, TemplateId id)
            where T : DTO.ISuccessModel
        {
            var response = await client.Get<T>(path);
            var model = objectService.ModelFrom(response.DataCollection.Rows.First());
            return GetTemplate(id, model);
        }

        // Simple custom JSON formatter just to make template as compact as possible.
        private string Format(DTO.DataObjectLite obj)
        {
            var buffer = new StringBuilder();

            using (var writer = new StringWriter(buffer))
            using (var indentedWriter = new IndentedTextWriter(writer, "  "))
            {
                Format(indentedWriter, obj, isRoot: true);
            }

            return buffer.ToString();
        }

        private string Format(IEnumerable<DTO.DataObjectLite> collection)
        {
            var buffer = new StringBuilder();
            buffer.Append('[');

            using (var writer = new StringWriter(buffer))
            using (var indentedWriter = new IndentedTextWriter(writer, "  "))
            {
                foreach (var obj in collection)
                {
                    Format(indentedWriter, obj, isRoot: false);
                }
            }

            buffer.Append(']');
            return buffer.ToString();
        }

        private void Format(IndentedTextWriter writer, DTO.DataObjectLite obj, bool isRoot)
        {
            writer.WriteLine("{");
            writer.Indent++;

            writer.WriteLine("Attributes: {");
            writer.Indent++;

            foreach (var attributeEntry in obj.Attributes)
            {
                Format(writer, attributeEntry.Key, attributeEntry.Value);
            }

            writer.Indent--;
            writer.WriteLine("},");

            Format(writer, obj.ChildObjects);
            writer.WriteLine($"SubclassId: \"{obj.SubclassId}\",");
            writer.Indent--;

            if (isRoot)
            {
                writer.WriteLine("}");
            }
            else
            {
                writer.WriteLine("},");
            }
        }

        private void Format(IndentedTextWriter writer, string attrId, DTO.Attribute attr)
        {
            writer.Write($"{attrId}: {{ ");
            bool hasValue = false;

            if (attr.Value != null)
            {
                hasValue = true;
                writer.Write($"value: \"{Escape(attr.Value)}\"");
            }

            if (!string.IsNullOrEmpty(attr.AliasValue))
            {
                if (hasValue)
                {
                    writer.Write(", ");
                }

                writer.Write($"aliasValue: \"{Escape(attr.AliasValue)}\" ");
            }

            writer.WriteLine("},");
        }

        private void Format(IndentedTextWriter writer, IEnumerable<DTO.DataObjectLiteCollection> childObjects)
        {
            if (childObjects != null && childObjects.Any())
            {
                writer.WriteLine("ChildObjects: [");

                foreach (var collection in childObjects)
                {
                    Format(writer, collection);
                }

                writer.WriteLine("],");
            }
        }

        private void Format(IndentedTextWriter writer, DTO.DataObjectLiteCollection collection)
        {
            writer.WriteLine("{");
            writer.Indent++;

            writer.WriteLine($"Id: \"{collection.Id}\",");
            writer.WriteLine($"ObjectId: \"{collection.ObjectId}\",");
            writer.WriteLine("Rows: [");

            foreach (var row in collection.Rows)
            {
                Format(writer, row, isRoot: false);
            }

            writer.WriteLine("],");
            writer.Indent--;
            writer.WriteLine("},");
        }

        private string Escape(string value)
        {
            return value.Replace("\"", "\\\"");
        }
    }
}
