﻿namespace SampleApp.Services
{
    using System;
    using System.IO;
    using System.Security.Cryptography;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// Represents persistent storage for refresh tokens.
    /// </summary>
    public class TokenStore
    {
        private const string FolderName = "SampleApp";
        private static byte[] Secret = Encoding.UTF8.GetBytes("S@mp1eApp!SeCr3t");
        private string fileName;
        private ILog logger;

        public TokenStore(CiamSettings settings, ILog logger)
        {
            // Sample app could use different CIAM configurations,
            // so to be able to distinguish tokens and avoid conflicts
            // a hash based on the key CIAM properties is used as a token file name.
            // Could be a constant if such kind of variety is not required.
            this.fileName = GetHash(settings) + ".token";
            this.logger = logger;
        }

        public async Task SaveToken(string token)
        {
            if (!string.IsNullOrEmpty(token))
            {
                var filePath = GetFilePath(ensureDir: true);

                using (var aes = Aes.Create())
                {
                    aes.Key = Secret;

                    using (var stream = File.OpenWrite(filePath))
                    using (var crypto = new CryptoStream(stream, aes.CreateEncryptor(), CryptoStreamMode.Write))
                    using (var writer = new StreamWriter(crypto))
                    {
                        // Save initialization vector.
                        await stream.WriteAsync(aes.IV, 0, aes.IV.Length);
                        await writer.WriteAsync(token);
                    }
                }
            }
        }

        public async Task<string> LoadToken()
        {
            var filePath = GetFilePath();

            try
            {
                if (File.Exists(filePath))
                {
                    return await LoadToken(filePath);
                }
            }
            catch (Exception ex)
            {
                logger.Log("Faild to load CIAM token:" + ex.Message);
            }

            return null;
        }

        private async Task<string> LoadToken(string filePath)
        {
            using (var aes = Aes.Create())
            using (var stream = File.OpenRead(filePath))
            {
                // Load initialization vector.
                var iv = new byte[aes.IV.Length];
                stream.Read(iv, 0, iv.Length);

                using (var crypto = new CryptoStream(stream, aes.CreateDecryptor(Secret, iv), CryptoStreamMode.Read))
                using (var reader = new StreamReader(crypto))
                {
                    return await reader.ReadToEndAsync();
                }
            }
        }

        private string GetFilePath(bool ensureDir = false)
        {
            // Use current user app data direcotry as a root.
            var dir = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), FolderName);

            if (ensureDir && !Directory.Exists(dir))
            {
                Directory.CreateDirectory(dir);
            }

            return Path.Combine(dir, fileName);
        }

        private string GetHash(CiamSettings settings)
        {
            var parts = new[] {
                Encoding.UTF8.GetBytes(settings.Audience),
                Encoding.UTF8.GetBytes(settings.Authority),
                Encoding.UTF8.GetBytes(settings.ClientId),
                Encoding.UTF8.GetBytes(settings.Scope),
            };

            var key = new byte[parts.Select(a => a.Length).Sum()];
            var offset = 0;

            foreach (var part in parts)
            {
                part.CopyTo(key, offset);
                offset += part.Length;
            }

            return GetHash(key);
        }

        private string GetHash(byte[] key)
        {
            using (var hashAlg = SHA256.Create())
            {
                var hash = hashAlg.ComputeHash(key);
                var buffer = new StringBuilder(hash.Length * 2);

                for (var i = 0; i < hash.Length; i++)
                {
                    buffer.Append($"{hash[i]:X2}");
                }

                return buffer.ToString();
            }
        }
    }
}
