﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using RestSharp;
using Newtonsoft.Json;
using SampleApp.Services;

namespace SampleApp.Views
{
    /// <summary>
    /// Base class implemented common logic for case code run and user interaction.
    /// </summary>
    public abstract partial class CaseBaseView : UserControl, IView, ILog
    {
        protected IRestClient client;
        protected ApiService apiService;
        protected DataObjectService objectService;
        protected TemplateService templateService;
        protected CacheService cacheService;

        public CaseBaseView(IRestClient client, AuthenticationService authService)
        {
            this.client = client;
            this.apiService = new ApiService(client, authService, this);
            this.objectService = new DataObjectService();
            this.templateService = new TemplateService(client, authService, this);
            this.cacheService = new CacheService(apiService, this);

            InitializeComponent();
        }

        public abstract string Caption { get; }

        public void Log(string message)
        {
            if (!string.IsNullOrEmpty(message))
            {
                textOutput.AppendText(message);
                textOutput.AppendText(Environment.NewLine);
            }
        }

        protected void LogErrors(DTO.DataObjectLiteCollection collection)
        {
            var errors = collection?.Errors;

            if (errors != null && errors.Count > 0)
            {
                var tab = new TabPage("Errors");
                var txt = new RichTextBox();
                txt.Dock = DockStyle.Fill;
                txt.ReadOnly = true;
                tab.Controls.Add(txt);
                tabOutput.TabPages.Add(tab);

                foreach (var error in errors)
                {
                    if (!string.IsNullOrEmpty(error.AttributeId))
                    {
                        txt.AppendText($"{error.AttributeId}: ");
                    }

                    txt.AppendText(error.Error?.Message);
                    txt.AppendText(Environment.NewLine);
                }
            }
        }

        protected abstract Task Execute();

        protected abstract void Reset();

        protected abstract Task Lookup(string key);

        // A few wrappers below to clean up inherited scenarios code a bit.

        /// <summary>
        /// Returns the Data Object ItemID value.
        /// </summary>
        protected string Get(DTO.DataObjectLite model, string attributeId) => objectService.Get(model, attributeId);

        /// <summary>
        /// Returns the Data Object attribute value converted to Int32.
        /// </summary>
        protected int GetInt32(DTO.DataObjectLite model, string attributeId) => objectService.GetInt32(model, attributeId);

        /// <summary>
        /// Returns the Data Object attribute value.
        /// </summary>
        protected Guid GetId(DTO.DataObjectLite model) => objectService.GetId(model);


        /// <summary>
        /// Updates the Data Object attribute value.
        /// </summary>
        protected void Set(DTO.DataObjectLite model, string attributeId, string value) => objectService.Set(model, attributeId, value);

        /// <summary>
        /// Creates full copy of the source Data Object provided applying the format function to its attributes.
        /// </summary>
        protected DTO.DataObjectLite Generate(DTO.DataObjectLite model, Func<string, string> format) => objectService.Generate(model, format);

        /// <summary>
        /// Creates full copy of the source Data Object filtering out NxItem and related(ex. ClientRel.Description) attributes.
        /// </summary>
        protected DTO.DataObjectLite Clone(DTO.DataObjectLite source) => objectService.Clone(source);

        /// <summary>
        /// Selects a list of all child objects corresponded to the children schema path(ex. 'Relate/Site').
        /// </summary>
        protected IList<DTO.DataObjectLite> Select(DTO.DataObjectLite source, string path) => objectService.Select(source, path);

        /// <summary>
        /// Creates formatter to replace %suffix% token with the actual value provided.
        /// </summary>
        protected Func<string, string> CreateFormatter(string suffix) => value => value.Replace(TemplateService.SuffixToken, suffix);

        /// <summary>
        /// Creates formatter to replace %suffix% and %date% tokens with the actual value provided.
        /// </summary>
        protected Func<string, string> CreateFormatter(string suffix, string date) =>
            value => value.Replace(TemplateService.SuffixToken, suffix).Replace(TemplateService.DateToken, date);

        /// <summary>
        /// Returns first index of the item matched the predicate or -1 if no one been found.
        /// </summary>
        protected int IndexOf(IList<DTO.DataObjectLite> items, Func<DTO.DataObjectLite, bool> predicate)
        {
            for (int i = 0; i < items.Count; i++)
            {
                if (predicate(items[i]))
                {
                    return i;
                }
            }

            return -1;
        }

        protected void SetLookupName(string name)
        {
            lLookup.Text = name + ":";
        }

        protected void AddInput(Control input)
        {
            input.Dock = DockStyle.Fill;
            panelInput.Controls.Add(input);
        }

        protected T AddInput<T>()
            where T : Control, new()
        {
            var ctrl = new T();
            AddInput(ctrl);
            return ctrl;
        }

        protected virtual void Clear()
        {
            textOutput.Clear();

            while (tabOutput.TabCount > 1)
            {
                tabOutput.TabPages.RemoveAt(1);
            }
        }

        protected virtual void DisableUI()
        {
            btnLookup.Enabled = false;
            btnSubmit.Enabled = false;
            btnReset.Enabled = false;
        }

        protected virtual void EnableUI()
        {
            btnLookup.Enabled = true;
            btnReset.Enabled = true;
            btnSubmit.Enabled = true;
        }

        private void btnReset_Click(object sender, EventArgs e)
        {
            try
            {
                Reset();
            }
            catch (Exception ex)
            {
                Log(ex.Message);
            }
        }

        private async void btnLookup_Click(object sender, EventArgs e)
        {
            DisableUI();

            try
            {
                await Lookup(tLookup.Text);
            }
            catch (Exception ex)
            {
                Log(ex.Message);
            }
            finally
            {
                EnableUI();
            }
        }

        private async void btnSubmit_Click(object sender, EventArgs e)
        {
            DisableUI();

            try
            {
                Clear();
                await Execute();
            }
            catch (Exception ex)
            {
                Log(ex.Message);

                if (ex is ApiDataException apiEx)
                {
                    LogErrors(apiEx.DataCollection);
                }
            }
            finally
            {
                EnableUI();
            }
        }
    }
}
