﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Net;
using System.Threading.Tasks;
using System.Windows.Forms;
using RestSharp;
using RestSharp.Serializers.NewtonsoftJson;
using SampleApp.Services;
using SampleApp.Views;

namespace SampleApp
{
    public partial class MainForm : Form, IViewController
    {
        private string overrideUrl;
        private IRestClient restClient;
        private ConfigurationService configurationService;
        private AuthenticationService authenticationService;
        private TemplateService templateService;
        private MainView mainView;

        public MainForm(string url = null)
        {
            this.overrideUrl = url;
            InitializeComponent();
        }


        /// <summary>
        /// Attaches or activates the view according to the view's code.
        /// </summary>
        public IView Attach(string code)
        {
            var tab = tabMain.TabPages[code];

            if (tab == null)
            {
                var view = CreateView(code);
                tab = new TabPage();
                tab.Name = code;
                tab.Text = view.Caption;
                ((Control)view).Dock = DockStyle.Fill;
                tab.Controls.Add(((Control)view));
                tabMain.TabPages.Add(tab);
            }

            tabMain.SelectedTab = tab;
            return (IView)tab.Controls[0];
        }

        /// <summary>
        /// Removes the view according to the view's code.
        /// </summary>
        public void Dettach(string code)
        {
            tabMain.TabPages.RemoveByKey(code);
        }

        /// <summary>
        /// Generates new view component according to the view's code.
        /// </summary>
        private IView CreateView(string code)
        {
            switch (code)
            {
                case ViewCodes.Main:
                    return new MainView(this);
                case ViewCodes.Login:
                    return new LoginView(configurationService);
                case ViewCodes.Matter:
                    return new MatterView(restClient, authenticationService);
                case ViewCodes.Client:
                    return new ClientView(restClient, authenticationService);
                case ViewCodes.Timekeeper:
                    return new TimekeeperView(restClient, authenticationService);
                case ViewCodes.EntityPerson:
                    return new EntityPersonView(restClient, authenticationService);
                case ViewCodes.EntityOrg:
                    return new EntityOrgView(restClient, authenticationService);
                case ViewCodes.Timecard:
                    return new TimecardView(restClient, authenticationService);
                case ViewCodes.TimecardPending:
                    return new TimecardPendingView(restClient, authenticationService);
                case ViewCodes.Costcard:
                    return new CostCardView(restClient, authenticationService);
                case ViewCodes.CostcardPending:
                    return new CostCardPendingView(restClient, authenticationService);
                case ViewCodes.Voucher:
                    return new VoucherView(restClient, authenticationService);
                case ViewCodes.ChargeCard:
                    return new ChargeCardView(restClient, authenticationService);
                case ViewCodes.ChargeCardPending:
                    return new ChargeCardPendingView(restClient, authenticationService);
                case ViewCodes.Payee:
                    return new PayeeView(restClient, authenticationService);
                case ViewCodes.VendorPayee:
                    return new VendorPayeeView(restClient, authenticationService);
                case ViewCodes.DirectCheck:
                    return new DirectChecksView(restClient, authenticationService);

                default:
                    throw new InvalidOperationException($"{code} view code is not implemented");
            }
        }

        private async void MainForm_Load(object sender, EventArgs e)
        {
            this.mainView = (MainView)Attach(ViewCodes.Main);

            try
            {
                mainView.UpdateState(enabled: false);
                this.configurationService = await CreateConfigurationService();
                await InitializeClient();
                await InitializeTemplates();
            }
            catch (Exception ex)
            {
                mainView.Log(ex.Message);
            }
            finally
            {
                mainView.UpdateState(enabled: true);
            }
        }

        /// <summary>
        /// Loads default Templates for each entity type supported.
        /// </summary>
        private async Task InitializeTemplates()
        {
            this.templateService = new TemplateService(restClient, authenticationService, mainView);
            await templateService.LoadTemplates();
        }

        private async Task InitializeClient()
        {
            var appSettings = configurationService.GetAppSettings();

            this.restClient = new RestClient(GetApiUrl(appSettings.Url));
            this.restClient.UseNewtonsoftJson();

            this.authenticationService = new AuthenticationService(configurationService, restClient, this, mainView);
            await authenticationService.Authenticate();

            mainView.ApiUrl = restClient.BaseUrl.AbsoluteUri;
        }

        private async Task<ConfigurationService> CreateConfigurationService()
        {
            if (overrideUrl != null)
            {
                mainView.Log("Retrieving server info...");

                var tmpClient = new RestClient(GetApiUrl(overrideUrl));
                var req = new RestRequest("info", Method.GET);
                var res = await tmpClient.ExecuteAsync<DTO.ServerInfo>(req);

                while (res.StatusCode == HttpStatusCode.Unauthorized && AuthenticationService.LoginWindows(tmpClient))
                {
                    res = await tmpClient.ExecuteAsync<DTO.ServerInfo>(req);
                }

                if (res.IsSuccessful)
                {
                    // Override authentication mode and api URL.
                    return new ConfigurationServiceEx(overrideUrl, res.Data);
                }

                // If no success just fallback to default configuration service.
                mainView.Log(res.ResponseStatus == ResponseStatus.Completed
                    ? $"{res.StatusCode} ({res.StatusDescription})"
                    : $"{res.ResponseStatus}. {res.ErrorMessage}");
            }

            return new ConfigurationService();
        }

        private string GetApiUrl(string url)
        {
            // Attach api version to the URL.
            // return url.EndsWith("/") ? url + "api/v1/" : url + "/api/v1/";
            return url;
        }
    }
}
