﻿namespace SampleApp.Services
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using RestSharp.Authenticators;
    using IdentityModel.OidcClient;
    using IdentityModel.OidcClient.Browser;
    using SampleApp.Views;

    /// <summary>
    /// Service responsible for 3E auhtentication routines.
    /// </summary>
    public class AuthenticationService
    {
        private AppSettings appSettings;
        private CiamSettings ciamSettings;
        private IRestClient restClient;
        private IViewController viewController;
        private ILog logger;
        private DateTime accessTokenExpiration = DateTime.MinValue;

        public AuthenticationService(
            ConfigurationService configurationService,
            IRestClient restClient,
            IViewController viewController,
            ILog logger)
        {
            this.restClient = restClient;
            this.viewController = viewController;
            this.logger = logger;
            this.appSettings = configurationService.GetAppSettings();

            if (this.appSettings.AuthenticationMode == AuthenticationMode.CIAM)
            {
                this.ciamSettings = configurationService.GetCiamSettings();
            }
        }

        /// <summary>
        /// Authenticates the REST client.
        /// </summary>
        /// <returns>The value indicating whether operation is completed successfully.</returns>
        public async Task<bool> Authenticate()
        {
            return appSettings.AuthenticationMode == AuthenticationMode.Windows
                ? AuthenticateWindows()
                : await AuthenticateCIAM();
        }

        public static bool LoginWindows(IRestClient client)
        {
            using (var form = new NtlmLoginForm())
            {
                if (form.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    client.Authenticator = new NtlmAuthenticator(form.UserName, form.Password);
                    return true;
                }
            }

            return false;
        }

        private bool AuthenticateWindows()
        {
            if (this.restClient.Authenticator == null)
            {
                // Use current windows identity as a default.
                this.restClient.Authenticator = new NtlmAuthenticator();
                return true;
            }

            // Allow to change user otherwise.
            return LoginWindows(this.restClient);
        }

        private async Task<bool> AuthenticateCIAM()
        {
            if (DateTime.Now < accessTokenExpiration)
            {
                // Token is not expired yet so no need to refresh.
                // If application still receives 401 means application configuration issue.
                return false;
            }

            var client = CreateOidcClient();
            var store = new TokenStore(ciamSettings, logger);
            return await AuthenticateCIAM(client, store);
        }

        private async Task<bool> AuthenticateCIAM(OidcClient client, TokenStore store)
        {
            var refreshToken = await store.LoadToken();

            if (refreshToken != null)
            {
                var result = await client.RefreshTokenAsync(refreshToken);

                if (!result.IsError)
                {
                    await store.SaveToken(result.RefreshToken);
                    return CompleteCIAM(result.AccessToken, result.AccessTokenExpiration);
                }

                logger.Log($"Failed to restore Access Token ({result.Error})");
            }

            // Refresh token is eithier not set (first run) or expired. 
            // User interaction is required in these cases.
            return await LoginCIAM(client, store);
        }

        private async Task<bool> LoginCIAM(OidcClient client, TokenStore store)
        {
            client.Options.Browser = (IBrowser)viewController.Attach(ViewCodes.Login);

            try
            {
                var result = await client.LoginAsync(CreateLoginRequest());

                if (result.IsError)
                {
                    logger.Log($"Failed to authenticate: {result.Error} {result.TokenResponse?.ErrorDescription}");
                    return false;
                }

                await store.SaveToken(result.RefreshToken);
                return CompleteCIAM(result.AccessToken, result.AccessTokenExpiration);
            }
            finally
            {
                viewController.Dettach(ViewCodes.Login);
            }
        }

        private bool CompleteCIAM(string accessToken, DateTime accessTokenExpiration)
        {
            this.restClient.Authenticator = new JwtAuthenticator(accessToken);
            this.accessTokenExpiration = accessTokenExpiration;
            return true;
        }

        private OidcClient CreateOidcClient()
        {
            var options = new OidcClientOptions
            {
                Authority = ciamSettings.Authority,
                ClientId = ciamSettings.ClientId,
                Scope = ciamSettings.Scope,
                RedirectUri = $"http://localhost:{ciamSettings.ReplyPort}/{ciamSettings.ReplyPath}"
            };

            return new OidcClient(options);
        }

        private LoginRequest CreateLoginRequest()
        {
            var loginRequest = new LoginRequest();

            loginRequest.FrontChannelExtraParameters = new Dictionary<string, string>()
            {
                ["login_hint"] = ciamSettings.LoginHint,
                ["connection"] = ciamSettings.Connection,
                ["response_mode"] = "query",
                ["audience"] = ciamSettings.Audience
            };

            return loginRequest;
        }
    }
}
