﻿namespace SampleApp.Views
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using SampleApp.Controls;
    using SampleApp.Services;

    public class ChargeCardView : CaseBaseView
    {
        ChargeCardInput input;

        public ChargeCardView(IRestClient client, AuthenticationService authService)
            : base(client, authService)
        {
            this.SetLookupName("Charge Index");
            this.input = new ChargeCardInput(this, this.objectService, this.cacheService);
            this.AddInput(input);
            this.Reset();
        }

        public override string Caption => "ChargeCard";

        protected override void Reset()
        {
            var template = templateService.GetTemplate(TemplateId.ChargeCard);
            input.SetChargeCardTemplate(template);
        }

        protected async override Task Lookup(string index)
        {
            if (!int.TryParse(index, out var chargeIndex) || chargeIndex <= 0)
            {
                Log("Please enter a valid Charge Index to copy from.");
                return;
            }

            var chargecard = await apiService.GetChargecard(chargeIndex);
            var model = Clone(chargecard);
            var template = templateService.GetTemplate(TemplateId.ChargeCard, model);
            input.SetChargeCardTemplate(template);
        }

        /// <summary>
        /// Generates ChargeCard list.
        /// </summary>
        protected override async Task Execute()
        {
            if (input.TryGetParameters(out var parameters))
            {
                var chargecards = new DTO.DataObjectLite[parameters.NumToCreate];
                var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);
                var matterIndex = 0;

                for (var i = 0; i < chargecards.Length; i++)
                {
                    var date = parameters.StartDate.AddDays(i / parameters.ChargeCardPerDay).ToString("yyyy-MM-dd");
                    chargecards[i] = Generate(parameters.ChargeCard, CreateFormatter(suffix.Increment(), date));

                    if (parameters.Matters != null)
                    {
                        Set(chargecards[i], "Matter", parameters.Matters[matterIndex++]);
                        matterIndex %= parameters.Matters.Count;
                    }
                }

                await apiService.Create<DTO.ChargeCardCreateResponse>(
                apiService.CreateGenericDataRequest("dataobject", chargecards, chargecards[0].SubclassId, "ChrgCardUpdate"));

                Log($"Total {parameters.NumToCreate} ChargeCard(s) successfully created.");
            }
        }
    }
}
