﻿
namespace SampleApp.Views
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using SampleApp.Controls;
    using SampleApp.Services;

    public class ClientView : CaseBaseView
    {
        private ClientInput input;

        public ClientView(IRestClient client, AuthenticationService authService)
            : base(client, authService)
        {
            this.SetLookupName("Client Number");
            this.input = new ClientInput(this, this.objectService);
            this.AddInput(input);
            this.Reset();
        }

        public override string Caption => "Client";

        protected override void Reset()
        {
            var template = templateService.GetTemplate(TemplateId.Client);
            input.SetClientTemplate(template);

            template = templateService.GetTemplate(TemplateId.EntityPerson);
            input.SetEntityTemplate(template);

            // empty matter list by default
            input.SetMattersTemplate("[]");
        }

        protected override async Task Lookup(string number)
        {
            if (string.IsNullOrWhiteSpace(number))
            {
                Log("Please enter a valid Client Number to copy from.");
                return;
            }

            var client = await apiService.GetClient(number);
            var model = Clone(client.AsObjectLite());
            var template = templateService.GetTemplate(TemplateId.Client, model);
            input.SetClientTemplate(template);

            var entIndex = GetInt32(model, "Entity");
            var entity = await apiService.GetEntity(entIndex);
            model = Clone(entity.AsObjectLite());

            template = string.Equals(entity.SubclassId, "EntityPerson", StringComparison.OrdinalIgnoreCase)
                ? templateService.GetTemplate(TemplateId.EntityPerson, model)
                : templateService.GetTemplate(TemplateId.EntityOrganization, model);

            input.SetEntityTemplate(template);

            var clientIndex = GetInt32(client.AsObjectLite(), "ClientIndex");
            var matters = await apiService.GetMattersByClient(clientIndex);
            var mattersModel = matters.Select(matter => Clone(matter.AsObjectLite()));
            template = templateService.GetTemplateList(TemplateId.Matter, mattersModel);
            input.SetMattersTemplate(template);
        }

        /// <summary>
        /// Generates Client list and its dependencies.
        /// </summary>
        protected async override Task Execute()
        {
            if (input.TryGetParameters(out var parameters))
            {
                var clients = new DTO.DataObjectLite[parameters.NumToCreate];
                var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

                for (var i = 0; i < clients.Length; i++)
                {
                    clients[i] = Generate(parameters.Client, CreateFormatter(suffix.Increment()));
                }

                if (parameters.GenerateEntity)
                {
                    var invSite = Get(parameters.Client, "InvoiceSite");
                    var sites = Select(parameters.Entity, "relate/site");
                    var siteIndex = IndexOf(sites, site => Get(site, "SiteIndex") == invSite);
                    var entities = await GenerateEntities(parameters);

                    for (var i = 0; i < entities.Count; i++)
                    {
                        var entityId = GetId(entities[i]);
                        var entity = await apiService.GetEntity(entityId);
                        var entityIndex = Get(entity.AsObjectLite(), "EntIndex");
                        Set(clients[i], "Entity", entityIndex);

                        if (siteIndex >= 0)
                        {
                            sites = Select(entity.AsObjectLite(), "relate/site");
                            Set(clients[i], "InvoiceSite", Get(sites[siteIndex], "SiteIndex"));
                        }
                    }
                }

                var response = await apiService.Create<DTO.ClientCreateResponse>(
                    apiService.CreateDataRequest<DTO.ClientCreateRequest>("client", clients));

                Log($"Total {parameters.NumToCreate} Client(s) successfully created.");

                if (parameters.GenerateMatters)
                {
                    await GenerateMatters(parameters, response.DataCollection.Rows.ToList());
                }
            }
        }

        protected async Task<List<DTO.DataObjectLite>> GenerateEntities(ClientParameters parameters)
        {
            var entities = new DTO.DataObjectLite[parameters.NumToCreate];
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

            for (var i = 0; i < entities.Length; i++)
            {
                entities[i] = Generate(parameters.Entity, CreateFormatter(suffix.Increment()));
            }

            var request = string.Equals(parameters.Entity.SubclassId, "EntityPerson", StringComparison.OrdinalIgnoreCase)
                ? apiService.CreateDataRequest<DTO.EntityCreateRequest>("entity/person", entities)
                : apiService.CreateDataRequest<DTO.EntityCreateRequest>("entity/organization", entities);

            var response = await apiService.Create<DTO.EntityCreateResponse>(request);

            Log($"Total {parameters.NumToCreate} Entity(s) successfully created.");
            return response.DataCollection.Rows.ToList();
        }

        protected async Task GenerateMatters(ClientParameters parameters, List<DTO.DataObjectLite> clients)
        {
            var matters = new List<DTO.DataObjectLite>();
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

            for (var i = 0; i < clients.Count; i++)
            {
                var clientId = GetId(clients[i]);
                var client = await apiService.GetClient(clientId);
                var clientIndex = Get(client.AsObjectLite(), "ClientIndex");

                for (var j = 0; j < parameters.Matters.Length; j++)
                {
                    var matter = Generate(parameters.Matters[j], CreateFormatter(suffix.Increment()));
                    Set(matter, "Client", clientIndex);
                    matters.Add(matter);
                }
            }

            var res = await apiService.Create<DTO.MatterCreateResponse>(
                    apiService.CreateDataRequest<DTO.MatterCreateRequest>("matter", matters));

            Log($"Total {matters.Count} Matter(s) successfully created.");

            await this.cacheService.CacheMatters(res.DataCollection.Rows.ToList());
        }
    }
}
