﻿namespace SampleApp.Views
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using SampleApp.Controls;
    using SampleApp.Services;

    public class CostCardView : CaseBaseView
    {
        CostcardInput input;

        public CostCardView(IRestClient client, AuthenticationService authService)
            : base(client, authService)
        {
            this.SetLookupName("Cost Index");
            this.input = new CostcardInput(this, this.objectService, this.cacheService);
            this.AddInput(input);
            this.Reset();
        }

        public override string Caption => "CostCard";

        protected override void Reset()
        {
            var template = templateService.GetTemplate(TemplateId.CostCard);
            input.SetCostCardTemplate(template);
        }

        protected async override Task Lookup(string index)
        {
            if (!int.TryParse(index, out var costIndex) || costIndex <= 0)
            {
                Log("Please enter a valid Cost Index to copy from.");
                return;
            }

            var costcard = await apiService.GetCostcard(costIndex);
            var model = Clone(costcard.AsObjectLite());
            var template = templateService.GetTemplate(TemplateId.CostCard, model);
            input.SetCostCardTemplate(template);
        }

        /// <summary>
        /// Generates CostCard list.
        /// </summary>
        protected override async Task Execute()
        {
            if (input.TryGetParameters(out var parameters))
            {
                var costcards = new DTO.DataObjectLite[parameters.NumToCreate];
                var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);
                var matterIndex = 0;

                for (var i = 0; i < costcards.Length; i++)
                {
                    var date = parameters.StartDate.AddDays(i / parameters.CostCardsPerDay).ToString("yyyy-MM-dd");
                    costcards[i] = Generate(parameters.CostCard, CreateFormatter(suffix.Increment(), date));

                    if (parameters.Matters != null)
                    {
                        Set(costcards[i], "Matter", parameters.Matters[matterIndex++]);
                        matterIndex %= parameters.Matters.Count;
                    }
                }

                await apiService.Create<DTO.CostCardCreateResponse>(
                    apiService.CreateDataRequest<DTO.CostCardCreateRequest>("cost/posted", costcards));

                Log($"Total {parameters.NumToCreate} CostCard(s) successfully created.");
            }
        }
    }
}
