﻿namespace SampleApp.Views
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using SampleApp.Controls;
    using SampleApp.Services;

    public class MatterView : CaseBaseView
    {
        private MatterInput input;

        public MatterView(IRestClient client, AuthenticationService authService)
            : base(client, authService)
        {
            this.SetLookupName("Matter Number");
            this.input = new MatterInput(this, this.objectService);
            this.AddInput(input);
            this.Reset();
        }

        public override string Caption => "Matter";

        protected override void Reset()
        {
            var template = templateService.GetTemplate(TemplateId.Matter);
            input.SetMatterTemplate(template);

            template = templateService.GetTemplate(TemplateId.Client);
            input.SetClientTemplate(template);

            template = templateService.GetTemplate(TemplateId.EntityOrganization);
            input.SetEntityTemplate(template);
        }

        protected override async Task Lookup(string number)
        {
            if (string.IsNullOrWhiteSpace(number))
            {
                Log("Please enter a valid Matter Number to copy from.");
                return;
            }

            var matter = await apiService.GetMatter(number);
            var model = Clone(matter.AsObjectLite());
            var template = templateService.GetTemplate(TemplateId.Matter, model);
            input.SetMatterTemplate(template);

            var clientIndex = GetInt32(model, "Client");
            var client = await apiService.GetClient(clientIndex);
            model = Clone(client.AsObjectLite());
            template = templateService.GetTemplate(TemplateId.Client, model);
            input.SetClientTemplate(template);

            var entIndex = GetInt32(model, "Entity");
            var entity = await apiService.GetEntity(entIndex);
            model = Clone(entity.AsObjectLite());

            template = string.Equals(entity.SubclassId, "EntityPerson", StringComparison.OrdinalIgnoreCase)
                ? templateService.GetTemplate(TemplateId.EntityPerson, model)
                : templateService.GetTemplate(TemplateId.EntityOrganization, model);

            input.SetEntityTemplate(template);
        }

        /// <summary>
        /// Generates Matter list and its dependencies.
        /// </summary>
        protected override async Task Execute()
        {
            if (input.TryGetParameters(out var parameters))
            {
                var matters = new DTO.DataObjectLite[parameters.NumToCreate];
                var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

                for (var i = 0; i < matters.Length; i++)
                {
                    matters[i] = Generate(parameters.Matter, CreateFormatter(suffix.Increment()));
                }

                if (parameters.GenerateClient)
                {
                    var clientsToCreate = (parameters.NumToCreate - 1) / parameters.MattersPerClient + 1;
                    var clients = await GenerateClients(parameters, clientsToCreate);
                    var clientIndex = string.Empty;
                    var lastId = Guid.Empty;

                    for (var i = 0; i < matters.Length; i++)
                    {
                        var clientId = GetId(clients[i / parameters.MattersPerClient]);

                        if (clientId != lastId)
                        {
                            // Load Client Index once moved to the next client.
                            var client = await apiService.GetClient(clientId);
                            clientIndex = Get(client.AsObjectLite(), "ClientIndex");
                            lastId = clientId;
                        }

                        Set(matters[i], "Client", clientIndex);
                    }
                }

                var res = await apiService.Create<DTO.MatterCreateResponse>(
                    apiService.CreateDataRequest<DTO.MatterCreateRequest>("matter", matters));

                Log($"Total {parameters.NumToCreate} Matter(s) successfully created.");

                await this.cacheService.CacheMatters(res.DataCollection.Rows.ToList());
            }
        }

        protected async Task<List<DTO.DataObjectLite>> GenerateClients(MatterParameters parameters, int clientsToCreate)
        {
            var clients = new DTO.DataObjectLite[clientsToCreate];
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

            for (var i = 0; i < clients.Length; i++)
            {
                clients[i] = Generate(parameters.Client, CreateFormatter(suffix.Increment()));
            }

            if (parameters.GenerateEntity)
            {
                var invSite = Get(parameters.Client, "InvoiceSite");
                var sites = Select(parameters.Entity, "relate/site");
                var siteIndex = IndexOf(sites, site => Get(site, "SiteIndex") == invSite);
                var entities = await GenerateEntities(parameters, clientsToCreate);

                for (var i = 0; i < entities.Count; i++)
                {
                    var entityId = GetId(entities[i]);
                    var entity = await apiService.GetEntity(entityId);
                    var entityIndex = Get(entity.AsObjectLite(), "EntIndex");
                    Set(clients[i], "Entity", entityIndex);

                    if (siteIndex >= 0)
                    {
                        sites = Select(entity.AsObjectLite(), "relate/site");
                        Set(clients[i], "InvoiceSite", Get(sites[siteIndex], "SiteIndex"));
                    }
                }
            }

            var response = await apiService.Create<DTO.ClientCreateResponse>(
                apiService.CreateDataRequest<DTO.ClientCreateRequest>("client", clients));

            Log($"Total {clientsToCreate} Client(s) successfully created.");
            return response.DataCollection.Rows.ToList();
        }

        protected async Task<List<DTO.DataObjectLite>> GenerateEntities(MatterParameters parameters, int entitiesToCreate)
        {
            var entities = new DTO.DataObjectLite[entitiesToCreate];
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

            for (var i = 0; i < entities.Length; i++)
            {
                entities[i] = Generate(parameters.Entity, CreateFormatter(suffix.Increment()));
            }

            var request = string.Equals(parameters.Entity.SubclassId, "EntityPerson", StringComparison.OrdinalIgnoreCase)
                ? apiService.CreateDataRequest<DTO.EntityCreateRequest>("entity/person", entities)
                : apiService.CreateDataRequest<DTO.EntityCreateRequest>("entity/organization", entities);

            var response = await apiService.Create<DTO.EntityCreateResponse>(request);

            Log($"Total {entitiesToCreate} Entity(s) successfully created.");
            return response.DataCollection.Rows.ToList();
        }
    }
}
