﻿namespace SampleApp.Views
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using RestSharp;
    using SampleApp.Controls;
    using SampleApp.Services;

    public class TimekeeperView : CaseBaseView
    {
        private TimekeeperInput input;

        public TimekeeperView(IRestClient client, AuthenticationService authService)
            : base(client, authService)
        {
            this.SetLookupName("Timekeeper Number");
            this.input = new TimekeeperInput(this, this.objectService);
            this.AddInput(input);
            this.Reset();
        }

        public override string Caption => "Timekeeper";

        protected override void Reset()
        {
            var template = templateService.GetTemplate(TemplateId.Timekeeper);
            input.SetTimekeeperTemplate(template);

            template = templateService.GetTemplate(TemplateId.EntityPerson);
            input.SetPersonTemplate(template);
        }

        protected override async Task Lookup(string number)
        {
            if (string.IsNullOrWhiteSpace(number))
            {
                Log("Please enter a valid Timekeeper Number to copy from.");
                return;
            }

            var timekeeper = await apiService.GetTimekeeper(number);
            var model = Clone(timekeeper.AsObjectLite());
            var template = templateService.GetTemplate(TemplateId.Timekeeper, model);
            input.SetTimekeeperTemplate(template);

            var personIndex = GetInt32(model, "Entity");
            var person = await apiService.GetEntityPerson(personIndex);
            model = Clone(person.AsObjectLite());
            template = templateService.GetTemplate(TemplateId.EntityPerson, model);
            input.SetPersonTemplate(template);
        }

        /// <summary>
        /// Generates Timekeeper list and its dependencies.
        /// </summary>
        protected override async Task Execute()
        {
            if (input.TryGetParameters(out var parameters))
            {
                var timekeepers = new DTO.DataObjectLite[parameters.NumToCreate];
                var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

                for (var i = 0; i < timekeepers.Length; i++)
                {
                    timekeepers[i] = Generate(parameters.Timekeeper, CreateFormatter(suffix.Increment()));
                }

                if (parameters.GeneratePerson)
                {
                    var persons = await GeneratePersons(parameters);

                    for (var i = 0; i < timekeepers.Length; i++)
                    {
                        var personId = GetId(persons[i]);
                        var person = await apiService.GetEntityPerson(personId);
                        var personIndex = Get(person.AsObjectLite(), "EntIndex");
                        Set(timekeepers[i], "Entity", personIndex);
                    }
                }

                await apiService.Create<DTO.TimekeeperCreateResponse>(
                    apiService.CreateDataRequest<DTO.TimekeeperCreateRequest>("timekeeper", timekeepers));

                Log($"Total {parameters.NumToCreate} Timekeeper(s) successfully created.");
            }
        }

        protected async Task<List<DTO.DataObjectLite>> GeneratePersons(TimekeeperParameters parameters)
        {
            var persons = new DTO.DataObjectLite[parameters.NumToCreate];
            var suffix = new Suffix(parameters.SuffixStart, parameters.SuffixFormat);

            for (var i = 0; i < persons.Length; i++)
            {
                persons[i] = Generate(parameters.Person, CreateFormatter(suffix.Increment()));
            }

            var response = await apiService.Create<DTO.EntityCreateResponse>(
                apiService.CreateDataRequest<DTO.EntityCreateRequest>("entity/person", persons));

            Log($"Total {parameters.NumToCreate} Person(s) successfully created.");
            return response.DataCollection.Rows.ToList();
        }
    }
}
